const https = require('https');
const mysql = require('mysql2');
const fs = require('fs');
const path = require('path')
const WebSocket = require('ws');
require('dotenv').config({ path: 'conecta.env' });

const poolMySQL = mysql.createPool({
    host: process.env.DB_MYSQL_SERVER,
    user: process.env.DB_MYSQL_USERNAME,
    password: process.env.DB_MYSQL_PASSWORD,
    database: process.env.DB_MYSQL_NAME
});

const server = https.createServer({
  cert: fs.readFileSync(path.join(__dirname, 'fullchain.pem')),
  key: fs.readFileSync(path.join(__dirname, 'privkey.pem')),
});
const wss = new WebSocket.Server({ server });

const clients = new Map(); 


server.listen(process.env.WS_PORT, () => {
    console.log(`Servidor WSS rodando na porta ${process.env.WS_PORT}`);
    console.log();
});

async function buscarPacientes(CNES, idPainel) {
    const query = `
        SELECT 
            t2.nome_paciente,
            CASE
                WHEN t1.tipo_atendimento_painel = 'N' THEN 'NORMAL'
                WHEN t1.tipo_atendimento_painel = 'P' THEN 'PRIORIDADE'
            END AS tipo_atendimento,
            t1.data_ultimaalteracao_statuspainel as dt_ultima_alteracao_status,
            t1.status_atendimento_painel AS status,
            t1.status_atendimento_painel_mostrar AS statusMostrar,
            status_atendimento_painel,
            UPPER(t4.nome_profissional) AS profissional,
            UPPER(t5.nome_setoratendimento) as nome_setoratendimento,
            UPPER(t5.comochamar_setor) as comochamar_setor
        FROM reg_tb_agendamentos_atender t1
        INNER JOIN tb_paciente t2 ON t1.id_paciente = t2.id_paciente
        RIGHT OUTER JOIN reg_tb_agenda t3 ON (t3.id_agenda = t1.id_agenda)
        INNER JOIN tb_profissional t4 ON t3.codigoprofissional_cnes = t4.codigoprofissional_cnes
        INNER JOIN reg_tb_setoratendimento t5 ON t4.id_setorlogado = t5.id_setoratendimento
        INNER JOIN tb_estabelecimento t6 ON t5.codigo_estabelecimento = t6.codigoestabelecimento_cnes
        INNER JOIN reg_tb_painelchamada_setor t7 ON t5.id_setoratendimento = t7.id_setoratendimento
        WHERE
         t1.status_atendimento_painel IN ('Chamando', 'Aguardando', 'Atendido', 'Em Atendimento')
         AND right(t6.cnes_estabelecimento,7) = ?
         AND DATE(t1.data_atendimento) = CURDATE()
         AND t7.id_painelchamada = ?`;
  return new Promise((resolve, reject) => {
    poolMySQL.query(query, [CNES, idPainel], (err, results) => {
      if (err) {
        console.error("Erro ao buscar dados:", err);
        return resolve({ error: "Erro ao executar consulta no banco de dados" });
      }

      const response = {
        em_atendimento: [],
        aguardando_atendimento: [],
        atendido: []
      };

      for (const row of results) {
        const pacienteComSala = {
          ...row,
          sala: `${row.comochamar_setor} ${row.nome_setoratendimento}`
        };

        switch (row.status) {
          case 'Chamando':
            response.em_atendimento.push(pacienteComSala);
            break;
          case 'Aguardando':
            response.aguardando_atendimento.push(pacienteComSala);
            break;
        }

        switch (row.statusMostrar) {
          case 'Atendido':
          case 'Em Atendimento':
          case 'Chamado':
            response.atendido.push(pacienteComSala);
            break;
        }

      }

      resolve(response);
    });
  });
}


wss.on('connection', (ws) => {
    console.log("Novo cliente conectado");
    let CNES = null;
    let idPainel = null;

    ws.on('message', (message) => {
        try {
            const { CNES: receivedCNES, idPainel: receivedPainel } = JSON.parse(message);

            if (!receivedCNES || !receivedPainel) {
                ws.send(JSON.stringify({ error: "CNES e/ou idPainel não fornecido(s)" }));
                return;
            }

            CNES = receivedCNES;
            idPainel = receivedPainel;
            clients.set(ws, { CNES, idPainel });


            buscarPacientes(CNES, idPainel).then(dados => {
                ws.send(JSON.stringify(dados));
            });

        } catch (error) {
            console.error("Erro ao processar mensagem:", error);
            ws.send(JSON.stringify({ error: "Formato de mensagem inválido" }));
        }
    });

    ws.on('close', () => {
    console.log("Cliente desconectado");
    clients.delete(ws); 
});

});


setInterval(() => {
    for (const [ws, { CNES, idPainel }] of clients.entries()) {
        buscarPacientes(CNES, idPainel)
            .then(dados => {
                if (ws.readyState === WebSocket.OPEN) {
                    ws.send(JSON.stringify(dados));
                }
            })
            .catch(err => {
                console.error("Erro ao buscar pacientes para cliente:", err);
            });
    }
}, 2000);


